# $Id: mt-uploaddir.pl 94 2025-12-15 02:17:53Z tajima $

package MT::Plugin::UploadDir;

use strict;
use MT::Plugin;
@MT::Plugin::UploadDir::ISA = qw(MT::Plugin);

use constant DETECT_FIELD => 0;

my $DEFAULT_EXT = <<"EOT";
audio:mp3,wma,m4a,midi,wav,aiff
videos:mp4,m4v,mpeg,avi,mov,wmv
images:jpe?g,png,gif,bmp
text:txt
docs:pdf,docx?,xlsx?,pptx?
src:pl,c,cc,rb,py,rs,go,js,css,html?,xml,php
archive:bz2,cab,gz,jar,lzh,rar,tar,taz,zip
EOT
my $PLUGIN_NAME = 'UploadDir';
my $VERSION = '0.90';

my $admin_theme_id = MT->config('AdminThemeId') || '';
my $DATA_PREFIX = 'data';
if (MT->version_number >= 9) {
    $DATA_PREFIX = 'data-bs';
}
elsif (MT->version_number >= 8) {
    $DATA_PREFIX = 'data-bs' if $admin_theme_id eq 'admin2023' or $admin_theme_id eq 'admin2025';
}
else {
    $DATA_PREFIX = 'data-bs' if $admin_theme_id eq 'bootstrap5';
}

use MT;
use MT::Util;

my $plugin = new MT::Plugin::UploadDir({
    name => $PLUGIN_NAME,
    version => $VERSION,
    description => "<MT_TRANS phrase='This plugin automatically switches the destination directory for the uploaded file by the file extension.'>",
    doc_link => 'https://labs.m-logic.jp/plugins/mt-uploaddir/docs/'.$VERSION.'/mt-uploaddir.html',
    author_name => 'M-Logic, Inc.',
    author_link => 'http://labs.m-logic.jp/',
    blog_config_template => \&template,
    l10n_class => 'UploadDir::L10N',
    settings => new MT::PluginSettings([
        ['upload_dir_enabled_default', { Default => 1, Scope => 'system' }],
        ['upload_dir_ext_list', { Default => $DEFAULT_EXT }],
        ['upload_dir_ext_default'],
    ]),
    schema_version => '1.0',
    system_config_template => \&system_template,
    registry => {
        object_types => {
            'blog' => {
                'enable_uploaddir' => 'integer meta',
            },
        },
        callbacks => {
            'MT::App::CMS::cms_pre_save.blog' => \&hdlr_cms_pre_save_blog,
            'MT::App::CMS::template_source.async_asset_upload' => \&hdlr_source_async_asset_upload_mt7,
            'MT::App::CMS::template_param.async_asset_upload' => \&hdlr_param_async_asset_upload_mt7,
            'MT::App::CMS::template_source.asset_upload_panel' => \&hdlr_source_async_asset_upload_mt7,
            'MT::App::CMS::template_param.asset_upload_panel' => \&hdlr_param_async_asset_upload_mt7,
            'MT::App::CMS::template_param.cfg_prefs' => \&hdlr_param_cfg_prefs_mt7,
            'MT::App::CMS::template_param.header' => \&hdlr_param_header,
        },
    },
});

if (MT->version_number >= 7) {
    if ($MT::DebugMode) {
        require MT::Util::Log; require Data::Dumper; MT::Util::Log->init();
    }
    MT->add_plugin($plugin);
}

sub instance { $plugin; }

sub system_template {
    my $tmpl = <<'EOT';
<mtapp:setting
    id="upload_dir_enabled_default"
    label="<__trans phrase="Enable:">">
    <p><input type="checkbox" value="1" name="upload_dir_enabled_default" id="upload_dir_enabled_default"<mt:if name="upload_dir_enabled_default"> checked="checked"</mt:if> class="cb"/> <label for="upload_dir_enabled_default"><__trans phrase="Enabled this plugin by default."></label></p>
</mtapp:setting>
EOT
    $tmpl;
}

sub template {
    my $tmpl;
    $tmpl = <<'EOT';
  <mtapp:setting
     id="upload_dir_ext_list"
     label="<__trans phrase="Extensions:">">
    <p><textarea class="form-control" name="upload_dir_ext_list" id="upload_dir_ext_list" cols="60" rows="10"><mt:var name="upload_dir_ext_list" escape="html"></textarea></p>
  </mtapp:setting>

  <mtapp:setting
     id="upload_dir_ext_default"
     label="<__trans phrase="Default directory:">">
    <input type="text" class="form-control text med" name="upload_dir_ext_default" id="upload_dir_ext_default" value="<mt:var name="upload_dir_ext_default" escape="html">" size="60" />
  </mtapp:setting>
EOT
    $tmpl;
}

sub save_config {
    my $plugin = shift;
    my ($param, $scope) = @_;
    if ($scope ne 'system') {
        my $app = MT->instance;
        unless (
            defined($plugin->build_text($app, $param->{'upload_dir_ext_list'})) && 
            defined($plugin->build_text($app, $param->{'upload_dir_ext_default'}))
        ) {
            my $msg = $plugin->translate('Error saving plugin settings: [_1]', $plugin->errstr);
            return $app->error($msg);
            die $msg;
        }
    }
    $plugin->SUPER::save_config($param, $scope);
}

sub enable_uploaddir {
    my ($plugin, $blog) = @_;

    my $enable = $blog->meta('enable_uploaddir');
    return 1 if $enable;
    return 0 if (!$enable && $enable eq '0');
    # default
    $plugin->get_config_value('upload_dir_enabled_default', 'system') ? 1 : 0;
}

sub upload_dir_ext_list {
    my $plugin = shift;
    my ($blog_id) = @_;
    my %plugin_param;

    $plugin->load_config(\%plugin_param, 'blog:'.$blog_id);
    my $key = $plugin_param{upload_dir_ext_list};
    unless ($key) {
        $plugin->load_config(\%plugin_param, 'system');
        $key = $plugin_param{upload_dir_ext_list};
    }
    $key;
}

sub upload_dir_ext_default {
    my $plugin = shift;
    my ($blog_id) = @_;
    my %plugin_param;

    $plugin->load_config(\%plugin_param, 'blog:'.$blog_id);
    my $key = $plugin_param{upload_dir_ext_default};
    unless ($key) {
        $plugin->load_config(\%plugin_param, 'system');
        $key = $plugin_param{upload_dir_ext_default};
    }
    $key;
}

sub build_text {
    my ($plugin, $app, $text) = @_;

    return '' unless $text;

    my $blog = $app->blog;
    my $author = $app->user;

    require MT::Template;
    require MT::Template::Context;
    my $tmpl = MT::Template->new;
    $tmpl->name('UploadDir Configuration');
    $tmpl->text($text);
    my $result = '';
    my $ctx = MT::Template::Context->new;
    $ctx->stash('author', $author);
    $ctx->stash('blog', $blog);
    $ctx->stash('blog_id', $blog->id);
    $tmpl->blog_id($blog->id);
    if ( $app->isa('MT::App::CMS') ) {
        my $mode = $app->param('__mode') || '';
        my $ctid;
        if (DETECT_FIELD) {
            my $content_f;
            my $custom_f;
            my $edit_field = $app->param('edit_field') || '';
            if ($edit_field) {
                if ($mode eq 'dialog_asset_modal') {
                    if ($edit_field =~ /\Aeditor-input-content-field-(\d+?)\z/) {
                        my $cfid = $1;
                        $content_f = MT->model('cf')->load($cfid);
                    }
                    if ($edit_field =~ /\Acustomfield_(.+?)\z/) {
                        my $cfbn = $1;
                        $custom_f = MT->model('field')->load({ basename => $cfbn });
                    }
                }
                elsif ($mode eq 'list_asset') {
                    if ($edit_field =~ /\Acustomfield_(.+?)\z/) {
                        my $cfbn = $1;
                        $custom_f = MT->model('field')->load({ basename => $cfbn });
                    }
                }
            }
            elsif ($mode eq 'list_asset') {
                my $cfid = $app->param('content_field_id') || '';
                if ($cfid =~ /\A\d+\z/) {
                    $content_f = MT->model('cf')->load($cfid);
                }
            }
            elsif ($mode eq 'view') {
                my $type = $app->param('_type') || '';
                if ($type eq 'content_data') {
                    $ctid = $app->param('content_type_id') || '';
                }
            }
            # 
            if ($content_f) {
                $ctx->stash('content_field_data', $content_f);
                $ctid = $content_f->content_type_id;
                my $cf_types = MT->registry('content_field_types');
                my $cf_type = $content_f->type || '';
                $ctx->var('content_field_id', $content_f->id);
                $ctx->var('content_field_name', $content_f->name);
                $ctx->var('content_field_description', $content_f->description);
                $ctx->var('content_field_unique_id', $content_f->unique_id);
                $ctx->var('content_field_type', $cf_type);
                $ctx->var('content_field_options', $content_f->options);
                my $cf_label = $cf_types->{$cf_type}{label};
                if ($cf_label && ref($cf_label) eq 'CODE') {
                    $cf_label = $cf_label->();
                }
                $ctx->var('content_field_type_label', $cf_label);
            }
            if ($custom_f) {
                $ctx->stash('field', $custom_f);
                $ctx->var('customfield_basename', $custom_f->basename);
            }
        }
        else {
            # DETECT_FIELD off
            if ($mode eq 'view' || $mode eq 'edit') {
                my $type = $app->param('_type') || '';
                if ($type eq 'content_data') {
                    $ctid = $app->param('content_type_id') || '';
                }
            }
            elsif ($mode eq 'dialog_asset_modal' || $mode eq 'dialog_list_asset') {
                my $edit_field = $app->param('edit_field') || '';
                if ($edit_field =~ /\Aeditor-input-content-field-(\d+?)\z/) {
                    my $cf = MT->model('cf')->load($1);
                    $ctid = $cf->content_type_id if $cf;
                }
            }
            elsif ($mode eq 'list_asset') {
                my $cfid = $app->param('content_field_id') || '';
                if ($cfid =~ /\A\d+\z/) {
                    my $cf = MT->model('cf')->load($cfid);
                    $ctid = $cf->content_type_id if $cf;
                }
            }
        }
        if ($ctid && $ctid =~ /\A\d+\z/) {
            my $ct = MT->model('content_type')->load($ctid);
            $ctx->stash('content_type', $ct) if $ct;
            ### temporary workaround for MT7.0
            my $cd = MT->model('cd')->new;
            $ctx->stash('content', $cd) if $cd;
        }
    }
    # $app->set_default_tmpl_params($tmpl);
    $result = $tmpl->build($ctx)
        or return $plugin->error($tmpl->errstr);
    $result;
}

# ex) images:bmp,jpe?g,gif,tiff?,png
sub parse_ext {
    my $plugin = shift;
    my $ext_list = shift;

    $ext_list =~ s/,/\|/g;

    my @lines = split(/\n/, $ext_list);

    my %ext = ();
    foreach my $line (@lines) {
        $line =~ s/[\r\n]//g;
        $line = MT::Util::trim($line);
        next if !$line;
        last if $line eq ':';
        if ($line =~ m/([^:]+):(.+)$/) {
            $ext{$2} = $1;
        }
    }

    %ext;
}

sub hdlr_cms_pre_save_blog {
    my ($cb, $app, $obj, $original) = @_;
    my $screen = $app->param('cfg_screen') or return 1;
    return 1 unless $screen eq 'cfg_prefs';
    my $enabled = $app->param('enable_uploaddir');
    $obj->enable_uploaddir($enabled ? 1 : 0);
    return 1;
}

# MT7+
sub make_setting_tmpl_mt7 {
    my ($app, $nofield) = @_;
    my $blog = $app->blog;
    my $blog_id = $blog->id;
    my $enable = $plugin->enable_uploaddir($blog);
    my $label = $plugin->translate('Subdirectories are determined by extensions.');
    my $display = $plugin->translate('Show configuration');
    my $hide = $plugin->translate('Hide configuration');
    my $cb_class = $DATA_PREFIX eq 'data-bs' ? 'form-check-input cb' : 'custom-control-input cb';
    my $ext_list = $plugin->build_text($app, $plugin->upload_dir_ext_list($blog_id));
    my $ext_default = $plugin->build_text($app, $plugin->upload_dir_ext_default($blog_id));
    my %ext = $plugin->parse_ext($ext_list);
    my $field = $nofield ? <<"HTML":
&nbsp;<span>($label <a href="javascript:void(0)" id="toggle_uploaddir" ${DATA_PREFIX}-toggle="collapse" ${DATA_PREFIX}-target="#uploaddir_extlist_panel" aria-expanded="false" aria-controls="uploaddir_extlist_panel" >$display</a>)</span>
HTML
<<"HTML";
<input type="checkbox" name="enable_uploaddir" id="enable_uploaddir" value="1" class="${cb_class}" />
<label for="enable_uploaddir" class="custom-control-label">$label</label>
&nbsp;<span>(<a href="javascript:void(0)" id="toggle_uploaddir" ${DATA_PREFIX}-toggle="collapse" ${DATA_PREFIX}-target="#uploaddir_extlist_panel" aria-expanded="false" aria-controls="uploaddir_extlist_panel">$display</a>)</span>
HTML

    my $initjs = <<"HTML";
jQuery(function() {
  jQuery('#enable_uploaddir').prop('checked', true);
  jQuery('#extra_path').addClass('disabled').attr('disabled', 'disabled');
});
HTML

    my $js = $nofield ? '' : <<"HTML";
jQuery('#enable_uploaddir').change( function() {
  if (jQuery(this).prop('checked')) {
    jQuery('#extra_path').addClass('disabled').attr('disabled', 'disabled');
  }
  else {
    jQuery('#extra_path').removeClass('disabled').removeAttr('disabled');
  }
});
jQuery('select#destination,select#upload_destination').change( function() {
  var map = jQuery(this).val();
  if (map == '') {
    setTimeout(function() {
      jQuery('.upload-extra-path').show();
    }, 0);
  }
});
HTML

    my $togglejs = <<"HTML";
jQuery('#uploaddir_extlist_panel').on('hidden.bs.collapse', function() {
    jQuery('#toggle_uploaddir').text('$display');
});
jQuery('#uploaddir_extlist_panel').on('shown.bs.collapse', function() {
    jQuery('#toggle_uploaddir').text('$hide');
});
HTML

    $js .= $initjs if $enable;
    $js .= $togglejs;
    my $table = '<thead>';
    $table .= '<tr><th class="dir">' . $plugin->translate('Directory') . '</th><th class="ext">' . $plugin->translate('Extensions') . "</th></tr>\n";
    $table .= "</thead><tbody>\n";
    foreach my $key ( keys %ext ) {
        $table .= '<tr><td class="dir">' . MT::Util::encode_html($ext{$key}) . '</td>';
        $key =~ s/\|/, /g;
        $table .= '<td class="ext">' . MT::Util::encode_html($key) . "</td></tr>\n";
    }
    $table .= '<tr><td class="dir default">' . MT::Util::encode_html($ext_default) . '</td><td class="ext default">' . $plugin->translate('Default') . "</td></tr>\n";
    $table .= "</tbody>\n";
    my $control_class = $nofield ? '' : ($DATA_PREFIX eq 'data-bs' ? 'form-check' : 'custom-control custom-checkbox');
    my $script = <<"HTML";
<div id="uploaddir_control" class="$control_class">
$field
<div id="uploaddir_extlist_panel" class="collapse"><table id="uploaddir_extlist">
$table
</table></div>
</div>
<mt:setvarblock name="jq_js_include" append="1">
$js
</mt:setvarblock>
HTML
    $script;
}

# MT7+
sub hdlr_source_async_asset_upload_mt7 {
    my ($eh, $app, $tmpl_ref) = @_;
    my $blog = $app->blog or return 1;
    my $blog_id = $blog->id;
    my $enable = $plugin->enable_uploaddir($blog);

    # replace hidden extra_path : allow_to_change_at_upload = false
    my $old = quotemeta('<input type="hidden" id="extra_path" name="extra_path" value="<mt:var name="extra_path" escape="html">" />');
    my $new = <<"HTML";
<input type="hidden" id="enable_uploaddir" name="enable_uploaddir" value="$enable" />
HTML
    $$tmpl_ref =~ s!($old)!$1\n$new!;
    my $label = $enable ? ' </div><div>' . make_setting_tmpl_mt7($app, 1) : '<mt:var name="extra_path" escape="html" />';
    # <mt:var name="upload_destination_label" escape="html">/<mt:var name="extra_path" escape="html">
    $old = quotemeta('<mt:var name="upload_destination_label"') . '[^\>]*?' . quotemeta('>/<mt:var name="extra_path"') . '[^\>]*?\>';
    $new = <<"HTML";
<mt:var name="upload_destination_label" escape="html" />/$label
HTML
    $$tmpl_ref =~ s!$old!$new!;

    $new = <<'HTML';
if (jQuery('input:checkbox#enable_uploaddir').prop('checked') || jQuery('input:hidden#enable_uploaddir').val() == 1 ) {
  const fileName = file.name || "";
  const match = fileName.match(/\.([^.]+)$/);
  const ext = match ? match[1].toLowerCase() : "";
  let targetDir = "";
  if (typeof uploaddir_config !== 'undefined' && ext) {
    for (const [exts, dir] of Object.entries(uploaddir_config.ext_map)) {
      const extPatterns = exts.split('|');
      const matched = extPatterns.some(pattern => {
        try {
          return ext.match(new RegExp('^' + pattern + '$', 'i'));
        } catch (e) {
          return ext.toLowerCase() === pattern.toLowerCase();
        }
      });
      if (matched) {
        targetDir = dir;
        break;
      }
    }
    if (!targetDir) {
      targetDir = uploaddir_config.ext_default || "";
    }
  }
  fd.append('extra_path', targetDir );
}
else {
  fd.append('extra_path', jQuery('#extra_path').val() );
}
HTML
    $old = quotemeta('fd.append(\'magic_token\', \'<mt:var name="magic_token">\');');
    $$tmpl_ref =~ s!($old)!$1\n$new!;

    $old = quotemeta('|| $fld.name === \'destination\'');
    $new = '|| $fld.name == \'extra_path\'';
    $$tmpl_ref =~ s!($old)!$1$new!;
    1;
}

# allow_to_change_at_upload = true
sub hdlr_param_async_asset_upload_mt7 {
    my ( $cb, $app, $param, $tmpl ) = @_;
    my $blog = $app->blog or return;
    MT::Util::Log->info('[UploadDir] allow_to_change_at_upload=' . (!defined $blog->allow_to_change_at_upload || $blog->allow_to_change_at_upload ? 'true' : 'false')) if $MT::DebugMode;
    return if defined $blog->allow_to_change_at_upload && !$blog->allow_to_change_at_upload;
    my $blog_id = $blog->id;
    my ($dest_node) = grep { $_->getAttribute('name') eq 'allow_to_change_at_upload' } @{ $tmpl->getElementsByTagName('if') };
    if ($dest_node) {
        require MT::Template;
        my $setting = make_setting_tmpl_mt7($app);
        my $setting_tmpl = MT::Template->new(
            type   => 'scalarref',
            source => \$setting
        );
        foreach my $token ( @{$setting_tmpl->tokens} ) {
            $tmpl->insertAfter($token, $dest_node);
            $dest_node = $token;
        }
    }
}

sub hdlr_param_cfg_prefs_mt7 {
    my ( $cb, $app, $param, $tmpl ) = @_;
    my $blog = $app->blog || return;
    $param->{enable_uploaddir} = $plugin->enable_uploaddir($blog);
    my $setting = make_setting_tmpl_mt7($app);
    my $dest_node = $tmpl->getElementById('upload-settings');
    if (!$dest_node) {
        MT::Util::Log->error('hdlr_param_cfg_prefs_mt7: upload-settings node not found') if $MT::DebugMode;
        return;
    }
    my $html = $dest_node->innerHTML();
    my $old = $DATA_PREFIX eq 'data-bs' ? quotemeta('<div class="form-check') : quotemeta('<div class="custom-control custom-checkbox');
    $html =~ s!($old)!$setting\n$1!;
    $dest_node->innerHTML($html);
}

sub hdlr_param_header {
    my ($cb, $app, $param, $tmpl) = @_;

    my $blog = $app->blog;
    return unless $blog;

    # Load and build the header template with proper context
    require MT::Template;
    require MT::Template::Context;

    my $lang_id = lc $app->current_language || 'en_us';
    my $blog_id = $blog->id;
    my $ext_list = $plugin->build_text($app, $plugin->upload_dir_ext_list($blog_id));
    my %ext_map = $plugin->parse_ext($ext_list);
    require JSON;
    my $uploaddir_config = {
        enabled => $plugin->enable_uploaddir($blog) ? $JSON::true : $JSON::false,
        ext_map => \%ext_map,
        ext_default => $plugin->build_text($app, $plugin->upload_dir_ext_default($blog_id)),
    };
    my $uploaddir_config_json = MT::Util::to_json($uploaddir_config);
    my $script = <<HTML;
<script type="text/javascript">
    var uploaddir_config = $uploaddir_config_json;
HTML
    if ($lang_id ne 'en_us') {
        my $phrases = [
            'Subdirectories are determined by extensions.',
            'Directory',
            'Extensions',
            'Show configuration',
            'Hide configuration',
        ];
        foreach my $phrase (@$phrases) {
            $script .= "    Lexicon['$phrase']='" . MT::Util::encode_js($plugin->translate($phrase)) . "';\n";
        }
    }
    $script .= "  </script>\n";

    if (MT->version_number >= 9) {
        # for AssetUploader
        my $filename = $MT::DebugMode ? "header.tmpl" : "header.min.tmpl";
        my $header_tmpl = MT::Template->new(
            type   => 'filename',
            source => $filename,
            path   => [ $plugin->template_paths ]
        );
        return $plugin->error(
            $app->translate(
                "Loading template '[_1]' failed: [_2]", $filename,
                MT::Template->errstr
            )
        ) unless defined $header_tmpl;
        $script .= $header_tmpl->text; # no MT Tags included
    }

    my $script_tmpl = MT::Template->new(
        type   => 'scalarref',
        source => \$script
    );
    my $js_includes = $tmpl->getElementsByName("js_include")
        or return;
    my $before = $js_includes->[0];
    my $tokens = $script_tmpl->tokens;
    foreach my $t (reverse @$tokens) {
        $tmpl->insertBefore($t, $before);
        $before = $t;
    }

    my $html_heads = $tmpl->getElementsByName("html_head") or return;
    my $before_css = $html_heads->[0];
    my $style = <<"HTML";
<style type="text/css">
#uploaddir_extlist {
    display: table;
    border-radius: 3px;
    border: 1px solid #c0c6c9;
    border-collapse: collapse;
    margin-bottom: 0;
}
#uploaddir_extlist th {
    text-align: left;
    background-color: #e6e6e6;
    border-bottom: 1px solid #c0c6c9;
}
#uploaddir_extlist th,
#uploaddir_extlist td {
    margin: 0;
    padding: 3px;
}
#uploaddir_extlist .dir {
    border-right: 1px solid #c0c6c9;
}
#uploaddir_extlist .default {
    border-top: 1px dotted #c0c6c9;
}
form#upload #uploaddir_control {
    display: block;
    vertical-align: baseline;
}
div#site_path-field div.field-content {
    float: left;
}
</style>
HTML
    my $css_tmpl = MT::Template->new(
        type   => 'scalarref',
        source => \$style
    );
    my $css_tokens = $css_tmpl->tokens;
    foreach my $t (reverse @$css_tokens) {
        $tmpl->insertBefore($t, $before_css);
        $before_css = $t;
    }
}

1;
